/*-----------------------------------------------------------------------------
   File: DDEx1.CPP

   Desc: Direct Draw example program 1.  Creates a Direct Draw 
         object and then a primary surface with a back buffer.
         The Program then creates a 3D object and rotates it while 
  		 performing the following -

			Z-Buffering
			Gouraud Shading
			Texture Mapping (Very Basic) Test texture is 12x12 

		 I have not used a very good palette system, so the gouraud 
		 shading is unfortunately limited to shades of one colour.
		 The Texture Mapping also has problems with multiple axes of 
		 rotation and the bottom few lines... Don't know why!!  All 
		 the Texture mapping was done without aid of any explanation
		 so please forgive how crapit is !!!!

		 Please note the code used to create this is based on parts 
		 from the following microsoft Direct Draw examples -
	
			DDEx1
			Flip2D
		
		 All the 3D code was written by me and the render code was 
		 significantly modified from "Flip2D" to provide the features 
		 above.  It no longer fills the triangles on the scan line. 
		 The Frame rate counter was also taken from "Flip2D".  The 
		 Fixed point data type was also taken from "Flip2D". All 
		 Windows initialisation code was taken from "DDEx1".

		 Sample frame rates on an AMD K6-2 300 with 64MB RAM and an 
		 8MB Savage3D at 640x480 (with vsync enabled).
			
			  Flat Shaded				- 59 fps* 		
			  Flat Shaded with Z-Buffer - 59 fps*
			  Gouraud Shaded**			- 59 fps*
			  Texture Mapped**			- 29 fps

		* The refresh rate, presumably, for 640x480 is 60 Hz.
		
		** Gouraud shading and texture mapping use the same fill routine even
		   if Z-Buffering is off.  They are just there to show what a 
		   difference a Z-Buffer makes.

			Oscar Forth June 1999
  -----------------------------------------------------------------------------*/

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif


//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "includes.h" // Contains all constant expressions.

//-----------------------------------------------------------------------------
// Local definitions
//-----------------------------------------------------------------------------

#define NAME                "OzDD"
#define TITLE               "Oz's Direct Draw Program"

//-----------------------------------------------------------------------------
// Default settings
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global data
//-----------------------------------------------------------------------------
LPDIRECTDRAW4               g_pDD = NULL;        // DirectDraw object
LPDIRECTDRAWSURFACE4        Primary = NULL;		 // DirectDraw primary surface
LPDIRECTDRAWSURFACE4        Back = NULL;		 // DirectDraw back surface
IDirectDrawPalette			*Palette;			 // DirectDraw Palette
BOOL                        g_bActive = FALSE;   // Is application active?


//-----------------------------------------------------------------------------
// Local data
//-----------------------------------------------------------------------------

// The Drawing Routines Z = Z-Buffered, G = Gouraud Shaded, T = Texture Mapped

void OzTriangle8(BYTE *p, TFace *Face);   // Flat Shaded no Z-Buffering
void OzTriangle8Z(BYTE *p, TFace *Face);  // Flat Shaded with Z-Buffering
void OzTriangle8ZG(BYTE *p, TFace *Face); // Gouraud Shaded with Z-Buffering
void OzTriangle8ZT(BYTE *p, TFace *Face); // Texture Mapped with Z-Buffering


void ClearZBuffer();					  // Clears the Z-Buffer

// The next six lines set up the 3D object structures.  They are not initialised here.

#define numverts 5
#define numedges 12
#define numfaces 4


T3dVertex *Vertices[numverts];
TEdge *Edges[numedges];
TFace *Faces[numfaces];

int move; // Used for rotating the shape.



//-----------------------------------------------------------------------------
// Name: ReleaseAllObjects()
// Desc: Finished with all objects we use; release them
//-----------------------------------------------------------------------------

static void
ReleaseAllObjects(void)
{
    if (g_pDD != NULL)
    {
        if (Primary != NULL)
        {
            Primary->Release();
            Primary = NULL;
        }
        g_pDD->Release();
        g_pDD = NULL;
    }
}




//-----------------------------------------------------------------------------
// Name: InitFail()
// Desc: This function is called if an initialization function fails
//-----------------------------------------------------------------------------
HRESULT
InitFail(HWND hWnd, HRESULT hRet, LPCTSTR szError,...)
{
    char                        szBuff[128];
    va_list                     vl;

    va_start(vl, szError);
    vsprintf(szBuff, szError, vl);
    ReleaseAllObjects();
    MessageBox(hWnd, szBuff, TITLE, MB_OK);
    DestroyWindow(hWnd);
    va_end(vl);
    return hRet;
}


// Use the blter to do a color fill to clear the back buffer
void SetSurfaceColour(DDBLTFX ddbltfx)
{
    Back->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);
}
//-----------------------------------------------------------------------------
// Name: UpdateFrame()
// Desc: Displays the proper text for the page
//-----------------------------------------------------------------------------

int FrameRate;
int FrameCount;
int FrameCount0;
DWORD FrameTime;
DWORD FrameTime0;

static void UpdateFrame()
{
	DDSURFACEDESC2 ddsd;
	ddsd.dwSize = sizeof(ddsd);
	
	HDC                         hdc; 
    DDBLTFX                     ddbltfx;  
	    
    
    
	ddbltfx.dwSize = sizeof(ddbltfx);
    ddbltfx.dwFillColor = 0;

	SetSurfaceColour(ddbltfx);
	
	// Increment then x, y and z angles.
	xang+=xstep; 
	yang+=ystep;
	zang+=zstep;
	
	
	if (Back->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL) == DD_OK) // Lock the back buffer
	{
		int loop = 0, loop2 = 0;
		ClearZBuffer(); // Clear the Zbuffer. 
		
		do {
			loop2 = 0;
			do {
					// Convert the 3d object to 2d.
					Faces[loop]->GetEdge(loop2)->GetFromVertex()->Convertto2d(xang,yang,zang, drawdist, prp, exaggerate);
					loop2++;			
			} while (loop2 < 3);

			// Use simple back face detection technique to save a bit of time. 

			if (Faces[loop]->Visible()) 
			{
				// Depending on the currently selected conditions draw the polgons.

				if (gouraud) OzTriangle8ZG((BYTE*)ddsd.lpSurface,Faces[loop]);
				else if (texture) OzTriangle8ZT((BYTE*)ddsd.lpSurface,Faces[loop]);
				else {

					if (!zbuf) OzTriangle8((BYTE*)ddsd.lpSurface,Faces[loop]);
					else OzTriangle8Z((BYTE*)ddsd.lpSurface,Faces[loop]);
				}					
			}
			loop++;
		} while (loop < numfaces);
		
		Back->Unlock(NULL); // Unlock the Back Buffer
	
	}
	
	// The following block displays all the relevant text.

	if (Back->GetDC(&hdc) == DD_OK) 
	{
		char Zmess[35];
		char Gourmess[35];
		char Exitmess[35];
		char Texmess[35];
		char Frate[35];
		int len,len2,len3,len4,len5;
	
		SetBkMode(hdc, TRANSPARENT);

		if (!zbuf) len = wsprintf(Zmess, "Hit 'Z' to turn Z Buffer on");
		else len = wsprintf(Zmess, "Hit 'Z' to turn Z Buffer off");
		if (!gouraud) len2 = wsprintf(Gourmess, "Hit 'G' to turn Gouraud shading on");
		else len2 = wsprintf(Gourmess, "Hit 'G' to turn Gouraud shading off");
		if (!texture) len3 = wsprintf(Texmess, "Hit 'T' to turn Texture mapping on");
		else len3 = wsprintf(Texmess, "Hit 'T' to turn Texture mapping off");
		len4 = wsprintf(Frate, "Frame rate : %d fps", FrameRate);
		len5 = wsprintf(Exitmess, "Hit [Escape] to exit");
		
		SetTextColor(hdc, RGB(255,255,0));
		
		TextOut(hdc, 0, 5, Zmess, len);
		TextOut(hdc, 0, 20, Gourmess, len2);
		TextOut(hdc, 0, 35, Texmess, len3);
		TextOut(hdc, 0, 65, Frate, len4);
		TextOut(hdc, 0, ScreenY-(ScreenX<640 ? 24:48),Exitmess,len5);
		Back->ReleaseDC(hdc);
	}
	
	// Calculate the frame rate.

	FrameCount++;
	FrameTime = timeGetTime();

	if (FrameTime - FrameTime0 > 1000)
	{
		FrameRate = (FrameCount - FrameCount0) * 1000 / (FrameTime - FrameTime0);
		FrameTime0 = FrameTime;
		FrameCount0 = FrameCount;
	}    

}	

//-----------------------------------------------------------------------------
// Name: WindowProc()
// Desc: The Main Window Procedure
//-----------------------------------------------------------------------------
long FAR PASCAL
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    HRESULT                     hRet;

    switch (message)
    {
        case WM_ACTIVATEAPP:
            g_bActive = (wParam == WA_ACTIVE) || (wParam == WA_CLICKACTIVE);
			return 0L;

        case WM_DESTROY:
            ReleaseAllObjects();
            PostQuitMessage(0);
            return 0L;

        case WM_KEYDOWN:
            switch (wParam)
            {
                case VK_ESCAPE:
                    PostMessage(hWnd, WM_CLOSE, 0, 0);
                    return 0L;
				
            }
            break;
		
		// Deal with the keypresses. i.e. change the render routines.	
		
		case WM_CHAR:
			switch (wParam)
			{
				case 90:
					zbuf = !zbuf;
					return 0L;
				case 122:
					zbuf = !zbuf;
					return 0L;
				case 71:
					gouraud = !gouraud;
					if (gouraud) 
					{
						texture = false;
					} 
					return 0L;
				case 103:
					gouraud = !gouraud;
					if (gouraud) 
					{
						texture = false;
					} 
					return 0L;
				case 84:
					texture = !texture;
					if (texture) 
					{
						gouraud = false;
					} 
					return 0L;
				case 116:
					texture = !texture;
					if (texture) 
					{
						gouraud = false;
					} 
					return 0L;
	
				case 45:
					drawdist-=10;
					return 0L;
				case 61:
					drawdist+=10;
					return 0L;

			}
			break;

		case WM_SETCURSOR:
            SetCursor(NULL);
            return TRUE;

    }
		    
		    
    return DefWindowProc(hWnd, message, wParam, lParam);
}

//-----------------------------------------------------------------------------
// Name: InitApp()
// Desc: Do work required for every instance of the application:
//          Create the window, initialize data
//-----------------------------------------------------------------------------
static HRESULT
InitApp(HINSTANCE hInstance, int nCmdShow)
{
    HWND                        hWnd;
    WNDCLASS                    wc;
    DDSURFACEDESC2              ddsd;
    DDSCAPS2                    ddscaps;
    HRESULT                     hRet;
    LPDIRECTDRAW                pDD;

    // Set up and register window class
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WindowProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_MAIN_ICON));
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH )GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName = NAME;
    wc.lpszClassName = NAME;
    RegisterClass(&wc);

    // Create a window
    hWnd = CreateWindowEx(WS_EX_TOPMOST,
                          NAME,
                          TITLE,
                          WS_POPUP,
                          0,
                          0,
                          GetSystemMetrics(SM_CXSCREEN),
                          GetSystemMetrics(SM_CYSCREEN),
                          NULL,
                          NULL,
                          hInstance,
                          NULL);
    
	
    if (!hWnd)
        return FALSE;
    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);
    SetFocus(hWnd);

    ///////////////////////////////////////////////////////////////////////////
    // Create the main DirectDraw object
    ///////////////////////////////////////////////////////////////////////////
    
	if (DirectDrawCreate(NULL, &pDD, NULL) != DD_OK)
		return InitFail(hWnd, hRet, "DirectDrawCreate FAILED");
    // Fetch DirectDraw4 interface
    if (pDD->QueryInterface(IID_IDirectDraw4, (LPVOID *) & g_pDD) != DD_OK)
        return InitFail(hWnd, hRet, "QueryInterface FAILED");

    // Get exclusive mode
    if (g_pDD->SetCooperativeLevel(hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN) != DD_OK)
        return InitFail(hWnd, hRet, "SetCooperativeLevel FAILED");

    
    if (g_pDD->SetDisplayMode(ScreenX, ScreenY, 8, 0, 0) != DD_OK)
        return InitFail(hWnd, hRet, "SetDisplayMode FAILED");

    // Create the primary surface with 1 back buffer
    ZeroMemory(&ddsd, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
        DDSCAPS_FLIP |
        DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount = 1;
    
	if (g_pDD->CreateSurface(&ddsd, &Primary, NULL) != DD_OK)
        return InitFail(hWnd, hRet, "CreateSurface FAILED");

    // Get a pointer to the back buffer
    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    if (Primary->GetAttachedSurface(&ddscaps, &Back) != DD_OK)
        return InitFail(hWnd, hRet, "GetAttachedSurface FAILED");
  
	PALETTEENTRY ape[256];
    
	HDC hdc = GetDC(NULL);
    
	if (GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE)
    {
        // get the current windows colors.
        GetSystemPaletteEntries(hdc, 0, 256, ape);

        // make a red, grn, and blu wash for our cube.
        /*for (int i=0; i<32; i++)
        {
            ape[10+64*0+i].peRed   = (i * 255/32);            
			ape[10+64*0+i].peGreen = 0;
            ape[10+64*0+i].peBlue  = 0;

            ape[10+64*1+i].peRed   = 0;
            ape[10+64*1+i].peGreen = i * 255/32;
            ape[10+64*1+i].peBlue  = 0;

            ape[10+64*2+i].peRed   = 0;
            ape[10+64*2+i].peGreen = 0;
            ape[10+64*2+i].peBlue  = i * 255/32;
        }*/
		for (int i=0; i<128; i++)
        {
            ape[10+i].peRed   = i * 255/130;            
			ape[10+i].peGreen = ((128 - i) * 255/130);
            ape[10+i].peBlue  = 0;

            
        }
				
        // create the palette.
        if (g_pDD->CreatePalette(DDPCAPS_8BIT, ape, &Palette, NULL) == DD_OK)
            Primary->SetPalette(Palette);
        
    }
    ReleaseDC(NULL, hdc);

    return DD_OK;
}




//-----------------------------------------------------------------------------
// Name: WinMain()
// Desc: Initialization, message loop
//-----------------------------------------------------------------------------
int PASCAL
WinMain(HINSTANCE hInstance,
        HINSTANCE hPrevInstance,
        LPSTR lpCmdLine,
        int nCmdShow)
{
	// Initialise the Polgon Vertex details

	Vertices[0] = new T3dVertex(   0, 100,   0,128,   0, 100);
	Vertices[1] = new T3dVertex(   0,-100, 100, 73, 100,-100);
	Vertices[2] = new T3dVertex(   0,-100,-100,103,-100,-100);
	Vertices[3] = new T3dVertex(-100,-100,   0, 63,-100,-100);
	Vertices[4] = new T3dVertex(100 ,-100,   0, 43, 100,-100);		    

	// Initialise the Polgon Edge details

	Edges[0] = new TEdge(Vertices[0],Vertices[2]);
	Edges[1] = new TEdge(Vertices[2],Vertices[1]);
	Edges[2] = new TEdge(Vertices[1],Vertices[0]);
	
	Edges[3] = new TEdge(Vertices[0],Vertices[1]);
	Edges[4] = new TEdge(Vertices[1],Vertices[2]);
	Edges[5] = new TEdge(Vertices[2],Vertices[0]);

	Edges[6] = new TEdge(Vertices[0],Vertices[3]);
	Edges[7] = new TEdge(Vertices[3],Vertices[4]);
	Edges[8] = new TEdge(Vertices[4],Vertices[0]);
	
	Edges[9] = new TEdge(Vertices[0],Vertices[4]);
	Edges[10] = new TEdge(Vertices[4],Vertices[3]);
	Edges[11] = new TEdge(Vertices[3],Vertices[0]);
	
	// Initialise the Polgon Face details

	Faces[0] = new TFace(Edges[0],Edges[1],Edges[2],73,Texture1, Tex1X, Tex1Y);
	Faces[1] = new TFace(Edges[3],Edges[4],Edges[5],137,Texture1, Tex1X, Tex2Y);
	Faces[2] = new TFace(Edges[6],Edges[7],Edges[8],201,Texture2, Tex2X, Tex2Y);
	Faces[3] = new TFace(Edges[9],Edges[10],Edges[11],158,Texture2, Tex2X, Tex2Y);


	MSG                         msg;

    if (InitApp(hInstance, nCmdShow) != DD_OK)
        return FALSE;
    while (TRUE)
    {
        if (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
        {
            if (!GetMessage(&msg, NULL, 0, 0))
                return msg.wParam;
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }

		// Run the program essentially.

        else if (g_bActive)
            {
                UpdateFrame();
                while (TRUE)
                {
                    HRESULT hRet;
					hRet = Primary->Flip(NULL, 0);
                    if (hRet == DD_OK)
                        break;
                    if (hRet == DDERR_SURFACELOST)
                    {
                        hRet = Primary->Restore();
                        if (hRet != DD_OK)
                            break;
                    }
                    if (hRet != DDERR_WASSTILLDRAWING)
                        break;
                }
            }
        else
        {
            WaitMessage();
        }
    }
	
}
/*********************************************************************************
 *																				 *
 *	The next two Procedures draw a filled polygon from the face colour details.	 *
 *  It iteratively follows down each edge of the polygon and draws a line		 *
 *	between the two points in the face colour.	OzTriangle8 makes extensive use	 *
 *  of the Fixed point data type defined in "Flip2D".							 *
 *																				 *
 *********************************************************************************/

void HLine(BYTE *p, int x0, int y, int x1,DWORD colour)
{
	int loop = 0;
	p += ((x0)+(ScreenX * y));

    do {
		
		*((DWORD*)p) = colour;
		loop++;
		p ++ ;
	} while (loop < (x1-x0));
}
void OzTriangle8(BYTE *p, TFace *Face)
{
   		
		{	
			int x1,y1,x2,y2,x3,y3;			
		{
			
			// Read in the Vertex details from the Face structure.
			
			T2dVertex *edge1f, *edge1t,*edge2f, *edge2t,*edge3f, *edge3t;
			{
				
				edge1f = Face->GetEdge1()->GetFrom2dVertex();
				edge1t = Face->GetEdge1()->GetTo2dVertex();
				
				edge2f = Face->GetEdge2()->GetFrom2dVertex();
				edge2t = Face->GetEdge2()->GetTo2dVertex();
				
			 	edge3f = Face->GetEdge3()->GetFrom2dVertex();
				edge3t = Face->GetEdge3()->GetTo2dVertex();
         		
			}
   			x1 = edge1f->GetX();
			y1 = edge1f->GetY();


			x2 = edge2f->GetX();
			y2 = edge2f->GetY();

			
			if (x2 == x1 && y2 == y1)
   			{
   				x2 = edge2t->GetX();
		   		y2 = edge2t->GetY();

   			}

			x3 = edge3f->GetX();
	   		y3 = edge3f->GetY();

	   		if ((x3 == x1 && y3 == y1) || (x3 == x2 && y3 == y2))
   			{
	   			x3 = edge3t->GetX();
	   			y3 = edge3t->GetY();

   			}
		}

		// Add an offset of half the Screen Width to the coords so that the polygon 
		// is centred.

		x1 = x1 + xo;
   		x2 = x2 + xo;
		x3 = x3 + xo;

   		y1 = y1 + yo;
		y2 = y2 + yo;
   		y3 = y3 + yo;
		  
		{
   			// Sort the Vertices such that (x1,y1) has the lowest y coord
			// (x2,y2) the next lowest and so on.

			int temp;
      		
			if (y2<y1)
			{
   	   			temp = y2;
      			y2 = y1;
         		y1 = temp;
				temp = x2;
   				x2 = x1;
      			x1 = temp;
		
			}
   			if (y3<y1)
      		{
      			temp = y3;
				y3 = y1;
   				y1 = temp;
				temp = x3;
         		x3 = x1;
				x1 = temp;
			}
   			if (y3<y2)
      		{
      			temp = y3;
				y3 = y2;
   				y2 = temp;
      			temp = x3;
         		x3 = x2;
				x2 = temp;

   			}
   		}
	DWORD colour = Face->GetColour(); // Read in the face colour.
	
	Fixed d = (Fixed)(x3 - x1) / (Fixed)(y3 - y1);
	    
	Fixed x  = x1;
    int y  = y1;

    Fixed d0 = (Fixed)(x2 - x1) / (Fixed)(y2 - y1);
	Fixed x0 = x1;
    if (d < d0)
    {
        while (y < y2)
        {

            HLine(p,x,y,x0,colour); // Draw the Horizontal Line 
            y++;
			x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y2)
        {
			HLine(p,x0,y,x,colour); // Draw the Horizontal Line 
            y++;
			x  += d;
            x0 += d0;
			
        }
    }

    d0 = (Fixed)(x3 - x2) / (Fixed)(y3 - y2);
	
    x0 = x2;
	
    if (x < x0)
    {
        while (y < y3)
        {

            HLine(p,x,y,x0,colour);	// Draw the Horizontal Line 
            y++;
			x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y3)
        {
	        
			HLine(p,x0,y,x,colour); // Draw the Horizontal Line 
            y++;
			x  += d;
            x0 += d0;
        }
    }
		
	}
		

}
/*********************************************************************************
 *																				 *
 *	The next Four Procedures do the same as above but now compare pixel 		 *
 *  positions against a  Z-Buffer.  This provides a system where by polygons are *
 *	clipped against each other, so no overlapping occurs.						 *
 *																				 *
 *********************************************************************************/

// Set the Z-Buffer to contain only 65535 the furthest away possible with a 16-bit
// Z-Buffer.

void ClearZBuffer()
{
	int loop =0;
	do {
		ZBuffer[loop]=65535; 
		loop++;
	} while(loop<ScreenSize);
	
}

// Write a coloured pixel at (x,y).

void SetPixel(BYTE *p, int x, int y, DWORD colour)
{
	p += ((x) + (ScreenX * y));
	*((DWORD*)p) = colour;
}

// This not only draws a line between two points but also checks the line against
// the Z-Buffer. 

void HLineZ(BYTE *p, int x0, int y0, int z0, int x1, int y1, int z1, DWORD colour)
{
	int scanpos = 0;
	int x = x0;
	Fixed z = z0;
	Fixed temp = -1;
	Fixed m = (Fixed)(z0 - z1) / (Fixed)(x0 - x1);
	if (y1 == y0) {
		do {
			scanpos = (x) + (ScreenX * y0);
			if ((int)z < -ZMin)  
			{
				// The following if block checks whether the value at that pixel 
				// position in the Z-Buffer is further away than the new one. 
				// If so it write the pixel.
				if (ZBuffer[scanpos] > ((int)z * -1)*((65536/ZRenderDepth)-1)) 
				{
					SetPixel(p,x,y0,colour);
					ZBuffer[scanpos] = ((int)z * -1)*((65536/ZRenderDepth)-1);
				}
			}
			// Exit if both ends are too close.
			else if (((int)z >= -ZMin) && (z1>=-ZMin)) return; 
			x++;
			z += m;
		} while (x<x1);
	}

}

// The differnece between this procedure and OZTriangle8 is that now a Z value is
// being calculated so that the Z-Buffer can be populated.

void OzTriangle8Z(BYTE *p, TFace *Face)
{
   		
		{	
			int x1,y1,z1,x2,y2,z2,x3,y3,z3;			
		{

			T2dVertex *edge1f, *edge1t,*edge2f, *edge2t,*edge3f, *edge3t;
			{
				
				edge1f = Face->GetEdge1()->GetFrom2dVertex();
				edge1t = Face->GetEdge1()->GetTo2dVertex();
				
				edge2f = Face->GetEdge2()->GetFrom2dVertex();
				edge2t = Face->GetEdge2()->GetTo2dVertex();
				
			 	edge3f = Face->GetEdge3()->GetFrom2dVertex();
				edge3t = Face->GetEdge3()->GetTo2dVertex();
         		
			}
   			x1 = edge1f->GetX();
			y1 = edge1f->GetY();
			z1 = Face->GetEdge1()->GetFrom3dVertex()->GetZ();

			x2 = edge2f->GetX();
			y2 = edge2f->GetY();
			z2 = Face->GetEdge2()->GetFrom3dVertex()->GetZ();
			
			if (x2 == x1 && y2 == y1)
   			{
   				x2 = edge2t->GetX();
		   		y2 = edge2t->GetY();
				z2 = Face->GetEdge2()->GetTo3dVertex()->GetZ();
   			}

			x3 = edge3f->GetX();
	   		y3 = edge3f->GetY();
			z3 = Face->GetEdge3()->GetFrom3dVertex()->GetZ();

	   		if ((x3 == x1 && y3 == y1) || (x3 == x2 && y3 == y2))
   			{
	   			x3 = edge3t->GetX();
	   			y3 = edge3t->GetY();
				z3 = Face->GetEdge3()->GetTo3dVertex()->GetZ();
   			}
		}
		x1 = x1 + xo;
   		x2 = x2 + xo;
		x3 = x3 + xo;

   		y1 = y1 + yo;
		y2 = y2 + yo;
   		y3 = y3 + yo;
  
		{
   			int temp;
      		if (y2<y1)
			{
   	   			temp = y2;
      			y2 = y1;
         		y1 = temp;
				temp = x2;
   				x2 = x1;
      			x1 = temp;
				temp = z2;
   				z2 = z1;
      			z1 = temp;
		
			}
   			if (y3<y1)
      		{
      			temp = y3;
				y3 = y1;
   				y1 = temp;
				temp = x3;
         		x3 = x1;
				x1 = temp;
				temp = z3;
         		z3 = z1;
				z1 = temp;
			}
   			if (y3<y2)
      		{
      			temp = y3;
				y3 = y2;
   				y2 = temp;
      			temp = x3;
         		x3 = x2;
				x2 = temp;
				temp = z3;
         		z3 = z2;
				z2 = temp;

   			}
   		}
	DWORD colour = Face->GetColour();
	Fixed d = (Fixed)(x3 - x1) / (Fixed)(y3 - y1);
	Fixed dz;
	if ((z3-z1) != 0) dz = (Fixed)(z3 - z1) / (Fixed)(y3 - y1);
	else dz = 0;
	
    
	Fixed x  = x1;
	Fixed z = z1;
    int y  = y1;
	Fixed d0z;
    Fixed d0 = (Fixed)(x2 - x1) / (Fixed)(y2 - y1);
    if ((z2-z1) != 0) d0z = (Fixed)(z2 - z1) / (Fixed)(y2 - y1);
	else d0z = 0;
    
	Fixed x0 = x1;
	Fixed z0 = z1;
    if (d < d0)
    {
        while (y < y2)
        {

            HLineZ(p,x,y,z,x0,y,z0,colour);
            y++;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y2)
        {
			HLineZ(p,x0,y,z0,x,y,z,colour);
            y++;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
			
        }
    }

    d0 = (Fixed)(x3 - x2) / (Fixed)(y3 - y2);
	if ((z3-z2) != 0) d0z = (Fixed)(z3 - z2) / (Fixed)(y3 - y2);
	else d0z = 0;
    x0 = x2;
	z0 = z2;
    if (x < x0)
    {
        while (y < y3)
        {

	        HLineZ(p,x,y,z,x0,y,z0,colour);
            y++;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y3)
        {
	        
			HLineZ(p,x0,y,z0,x,y,z,colour);
            y++;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }


	}


}

/*********************************************************************************
 *																				 *
 *	The next Two Procedures do the same as above bu now provide linear			 *
 *  interpolation of the colours (Gouraud Shading).  HLineZG uses the same		 *
 *	SetPixel procedure as above.												 *
 *																				 *
 *********************************************************************************/

void HLineZG(BYTE *p, int x0, int y0, int z0,Fixed c0, int x1, int y1, int z1, Fixed c1)
{
	int scanpos = 0;
	int x = x0;
	Fixed z = z0;
	Fixed c = c0;
	Fixed temp = -1;
	Fixed zm = (Fixed)(z0 - z1) / (Fixed)(x0 - x1);
	Fixed cm = (Fixed)(c0 - c1) / (Fixed)(x0 - x1);
	if (y1 == y0) {
		do {
			scanpos = (x) + (ScreenX * y0);
			if ((int)z < -ZMin)  
			{
				if ((ZBuffer[scanpos] > ((int)z * -1)*((65536/ZRenderDepth)-1)) && (zbuf))
				{
					SetPixel(p,x,y0,(int)c);
					ZBuffer[scanpos] = ((int)z * -1)*((65536/ZRenderDepth)-1);
				}
				else if (!zbuf) SetPixel(p,x,y0,(int)c);
			}
			else if (((int)z >= -ZMin) && (z1>=-ZMin)) return;
			x++;
			z += zm;
			c += cm;
		} while (x<x1);
	}

}

// Now not only is a Z value stored but also a colour value.  The Colour value is
// treated as another axis.

void OzTriangle8ZG(BYTE *p, TFace *Face)
{
   		
		{	
			int x1,y1,z1,c1,x2,y2,z2,c2,x3,y3,z3,c3;			
		{

			T2dVertex *edge1f, *edge1t,*edge2f, *edge2t,*edge3f, *edge3t;
			{
				
				edge1f = Face->GetEdge1()->GetFrom2dVertex();
				edge1t = Face->GetEdge1()->GetTo2dVertex();
				
				edge2f = Face->GetEdge2()->GetFrom2dVertex();
				edge2t = Face->GetEdge2()->GetTo2dVertex();
				
			 	edge3f = Face->GetEdge3()->GetFrom2dVertex();
				edge3t = Face->GetEdge3()->GetTo2dVertex();
         		
			}
   			x1 = edge1f->GetX();
			y1 = edge1f->GetY();
			z1 = Face->GetEdge1()->GetFrom3dVertex()->GetZ();
			c1 = Face->GetEdge1()->GetFrom3dVertex()->GetColour();

			x2 = edge2f->GetX();
			y2 = edge2f->GetY();
			z2 = Face->GetEdge2()->GetFrom3dVertex()->GetZ();
			c2 = Face->GetEdge2()->GetFrom3dVertex()->GetColour();

			if (x2 == x1 && y2 == y1)
   			{
   				x2 = edge2t->GetX();
		   		y2 = edge2t->GetY();
				z2 = Face->GetEdge2()->GetTo3dVertex()->GetZ();
				c2 = Face->GetEdge2()->GetTo3dVertex()->GetColour();
   			}

			x3 = edge3f->GetX();
	   		y3 = edge3f->GetY();
			z3 = Face->GetEdge3()->GetFrom3dVertex()->GetZ();
			c3 = Face->GetEdge3()->GetFrom3dVertex()->GetColour();

	   		if ((x3 == x1 && y3 == y1) || (x3 == x2 && y3 == y2))
   			{
	   			x3 = edge3t->GetX();
	   			y3 = edge3t->GetY();
				z3 = Face->GetEdge3()->GetTo3dVertex()->GetZ();
				c3 = Face->GetEdge3()->GetTo3dVertex()->GetColour();
   			}
		}
		x1 = x1 + xo;
   		x2 = x2 + xo;
		x3 = x3 + xo;

   		y1 = y1 + yo;
		y2 = y2 + yo;
   		y3 = y3 + yo;
  
		{
   			int temp;
      		if (y2<y1)
			{
   	   			temp = y2;
      			y2 = y1;
         		y1 = temp;
				temp = x2;
   				x2 = x1;
      			x1 = temp;
				temp = z2;
   				z2 = z1;
      			z1 = temp;
				temp = c2;
   				c2 = c1;
      			c1 = temp;
		
			}
   			if (y3<y1)
      		{
      			temp = y3;
				y3 = y1;
   				y1 = temp;
				temp = x3;
         		x3 = x1;
				x1 = temp;
				temp = z3;
         		z3 = z1;
				z1 = temp;
				temp = c3;
         		c3 = c1;
				c1 = temp;

			}
   			if (y3<y2)
      		{
      			temp = y3;
				y3 = y2;
   				y2 = temp;
      			temp = x3;
         		x3 = x2;
				x2 = temp;
				temp = z3;
         		z3 = z2;
				z2 = temp;
				temp = c3;
         		c3 = c2;
				c2 = temp;

   			}
   		}
	
	Fixed d = (Fixed)(x3 - x1) / (Fixed)(y3 - y1);
	Fixed dz;
	if ((z3-z1) != 0) dz = (Fixed)(z3 - z1) / (Fixed)(y3 - y1);
	else dz = 0;
	Fixed dc = (Fixed)(c3 - c1) / (Fixed)(y3 - y1);
	    
	Fixed x  = x1;
	Fixed z = z1;
	Fixed c = c1;

    int y  = y1;
	Fixed d0z;
    Fixed d0 = (Fixed)(x2 - x1) / (Fixed)(y2 - y1);
	
    if ((z2-z1) != 0) d0z = (Fixed)(z2 - z1) / (Fixed)(y2 - y1);
	else d0z = 0;
    Fixed d0c = (Fixed)(c2 - c1) / (Fixed)(y2 - y1);

	Fixed x0 = x1;
	Fixed z0 = z1;
	Fixed c0 = c1;
    if (d < d0)
    {
        while (y < y2)
        {

            HLineZG(p,x,y,z,c,x0,y,z0,c0);
            y++;
			c += dc;
			c0 += d0c;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y2)
        {
			HLineZG(p,x0,y,z0,c0,x,y,z,c);
            y++;
			c += dc;
			c0 += d0c;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
			
        }
    }

    d0 = (Fixed)(x3 - x2) / (Fixed)(y3 - y2);
	if ((z3-z2) != 0) d0z = (Fixed)(z3 - z2) / (Fixed)(y3 - y2);
	else d0z = 0;
	d0c = (Fixed)(c3 - c2) / (Fixed)(y3 - y2);
    x0 = x2;
	z0 = z2;
	c0 = c2;
    if (x < x0)
    {
        while (y < y3)
        {

	        HLineZG(p,x,y,z,c,x0,y,z0,c0);
            y++;
			c += dc;
			c0 += d0c;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y3)
        {
	        
			HLineZG(p,x0,y,z0,c0,x,y,z,c);
            y++;
			c += dc;
			c0 += d0c;
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }


	}


}
/*********************************************************************************
 *																				 *
 *	The next Two Procedures do the same as the Z-Buffer algorithm but now		 *
 *  map a texture on to the polygon surface.  Again this uses the same SetPixel  *
 *  routine.  N.B. The texture is skewed at certain angles.  					 *																		
 *																				 *
 *********************************************************************************/


void HLineZT(BYTE *p, int x0, int y0, int z0,int s0, int t0, int x1, int y1, int z1,int s1, int t1, DWORD *Texture, int TexX, int TexY)
{
	int scanpos = 0;
	int x = x0;
	
	Fixed s = s0;
	Fixed t = t0;
	
	Fixed z = z0;

	Fixed m = (Fixed)(z0 - z1) / (Fixed)(x0 - x1);

	Fixed ds = (Fixed)(s0 - s1) / (Fixed)(x0 - x1);

	Fixed dt = ((Fixed)(t0 - t1) / (Fixed)(s0 - s1)) * ds;


	int TexPosX = 0;
	int TexPosY = 0;
	
	if (y1 == y0) {
		do {
			scanpos = (x) + (ScreenX * y0);
			
			if ((int)s >= 0)
			{
				TexPosX = TexX - ((int)s % TexX) - 1; // Correct for positive x coords
				
			}
			else 
			{
				TexPosX = ((int)s % -TexX) * -1; // Correct for negative x coords
			}
			if ((int)t >= 0)
			{
				TexPosY = TexY - ((int)t % TexY) - 1; // Correct for positive y coords
			}
			else 
			{
				TexPosY = ((int)t  % -TexY) * -1; // Correct for negative y coords
			}
			
			
			
			if ((int)z < -ZMin)  
			{
				if ((ZBuffer[scanpos] > ((int)z * -1)*((65536/ZRenderDepth)-1)) && (zbuf))
				{
					// Write the correct Texel.
					SetPixel(p,x,y0,Texture[TexPosX + (TexPosY * TexX)]);
					ZBuffer[scanpos] = ((int)z * -1)*((65536/ZRenderDepth)-1);
				}
				else if (!zbuf) SetPixel(p,x,y0,Texture[TexPosX + (TexPosY * TexX)]);

			}
			else if (((int)z >= -ZMin) && (z1>=-ZMin)) return;
			x++;
			
			s += ds;
			t += dt;
			
			z += m;
		} while (x<x1);
	}

}

// Now instead of a colour axis we have two more s and t. These are traversed as if the
// polygon had not been rotated so that the texture orientation is correct.

void OzTriangle8ZT(BYTE *p, TFace *Face)
{
   		
		{	
			int x1,y1,z1,s1,t1,x2,y2,z2,s2,t2,x3,y3,z3,s3,t3;			
		{

			T2dVertex *edge1f, *edge1t,*edge2f, *edge2t,*edge3f, *edge3t;
			{
				
				edge1f = Face->GetEdge1()->GetFrom2dVertex();
				edge1t = Face->GetEdge1()->GetTo2dVertex();
				
				edge2f = Face->GetEdge2()->GetFrom2dVertex();
				edge2t = Face->GetEdge2()->GetTo2dVertex();
				
			 	edge3f = Face->GetEdge3()->GetFrom2dVertex();
				edge3t = Face->GetEdge3()->GetTo2dVertex();
         		
			}
   			x1 = edge1f->GetX();
			y1 = edge1f->GetY();
			z1 = Face->GetEdge1()->GetFrom3dVertex()->GetZ();
			s1 = Face->GetEdge1()->GetFromVertex()->GetS();
			t1 = Face->GetEdge1()->GetFromVertex()->GetT();

			x2 = edge2f->GetX();
			y2 = edge2f->GetY();
			z2 = Face->GetEdge2()->GetFrom3dVertex()->GetZ();
			s2 = Face->GetEdge2()->GetFromVertex()->GetS();
			t2 = Face->GetEdge2()->GetFromVertex()->GetT();

			if (x2 == x1 && y2 == y1)
   			{
   				x2 = edge2t->GetX();
		   		y2 = edge2t->GetY();
				z2 = Face->GetEdge2()->GetTo3dVertex()->GetZ();
				s2 = Face->GetEdge2()->GetToVertex()->GetS();
				t2 = Face->GetEdge2()->GetToVertex()->GetT();
  			}

			x3 = edge3f->GetX();
	   		y3 = edge3f->GetY();
			z3 = Face->GetEdge3()->GetFrom3dVertex()->GetZ();
			s3 = Face->GetEdge3()->GetFromVertex()->GetS();
			t3 = Face->GetEdge3()->GetFromVertex()->GetT();

	   		if ((x3 == x1 && y3 == y1) || (x3 == x2 && y3 == y2))
   			{
	   			x3 = edge3t->GetX();
	   			y3 = edge3t->GetY();
				z3 = Face->GetEdge3()->GetTo3dVertex()->GetZ();
				s3 = Face->GetEdge3()->GetToVertex()->GetS();
				t3 = Face->GetEdge3()->GetToVertex()->GetT();

   			}
		}
		x1 = x1 + xo;
   		x2 = x2 + xo;
		x3 = x3 + xo;

   		y1 = y1 + yo;
		y2 = y2 + yo;
   		y3 = y3 + yo;
  
		{
   			int temp;
      		if (y2<y1)
			{
   	   			temp = y2;
      			y2 = y1;
         		y1 = temp;
				temp = x2;
   				x2 = x1;
      			x1 = temp;
				temp = z2;
   				z2 = z1;
      			z1 = temp;
				
				temp = s2;
   				s2 = s1;
      			s1 = temp;
				temp = t2;
   				t2 = t1;
      			t1 = temp;
				
			}
   			if (y3<y1)
      		{
      			temp = y3;
				y3 = y1;
   				y1 = temp;
				temp = x3;
         		x3 = x1;
				x1 = temp;
				temp = z3;
         		z3 = z1;
				z1 = temp;
				
				temp = s3;
				s3 = s1;
   				s1 = temp;
				temp = t3;
         		t3 = t1;
				t1 = temp;
			}
   			if (y3<y2)
      		{
      			temp = y3;
				y3 = y2;
   				y2 = temp;
      			temp = x3;
         		x3 = x2;
				x2 = temp;
				temp = z3;
         		z3 = z2;
				z2 = temp;
				
				temp = s3;
				s3 = s2;
   				s2 = temp;
      			temp = t3;
         		t3 = t2;
				t2 = temp;


   			}
   		}
	
	Fixed d = (Fixed)(x3 - x1) / (Fixed)(y3 - y1);
	Fixed dz;
	
	
	
	if ((z3-z1) != 0) dz = (Fixed)(z3 - z1) / (Fixed)(y3 - y1);
	else dz = 0;
	
    
	Fixed ds = (Fixed)(s3 - s1) / (Fixed)(y3 - y1);
	Fixed dt = ((Fixed)(t3 - t1) / (Fixed)(s3 - s1)) * ds;

	Fixed x  = x1;
	Fixed z = z1;
	
	Fixed s = s1;
	Fixed t = t1;
    
	int y  = y1;
	
	Fixed d0z;
    
	Fixed d0 = (Fixed)(x2 - x1) / (Fixed)(y2 - y1);
    if ((z2-z1) != 0) d0z = (Fixed)(z2 - z1) / (Fixed)(y2 - y1);
	else d0z = 0;
    
	
	Fixed d0s = (Fixed)(s2 - s1) / (Fixed)(y2 - y1);
	Fixed d0t = ((Fixed)(t2 - t1) / (Fixed)(s2 - s1)) * d0s;
	
	Fixed x0 = x1;
	Fixed z0 = z1;
	
	Fixed s0 = s1;
	Fixed t0 = t1;
    
	if (d < d0)
    {
        while (y < y2)
        {

            HLineZT(p,x,y,z,s,t,x0,y,z0,s0,t0,Face->GetTexture(),Face->GetTexX(),Face->GetTexY());
          
			y++;
			
			t += dt;
			t0 += d0t;
		
			s += ds;
			s0 += d0s;

			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y2)
        {
			HLineZT(p,x0,y,z0,s0,t0,x,y,z,s,t,Face->GetTexture(),Face->GetTexX(),Face->GetTexY());

			y++;
			
			t += dt;
			t0 += d0t;
			
			s += ds;
			s0 += d0s;
			
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
			
        }
    }

    d0 = (Fixed)(x3 - x2) / (Fixed)(y3 - y2);
	if ((z3-z2) != 0) d0z = (Fixed)(z3 - z2) / (Fixed)(y3 - y2);
	else d0z = 0;
    
	d0s = (Fixed)(s3 - s2) / (Fixed)(y3 - y2);
	d0t = ((Fixed)(t3 - t2) / (Fixed)(s3 - s2)) * d0s;

	x0 = x2;
	z0 = z2;
	s0 = s2;
	t0 = t2;
    
    if (x < x0)
    {
        while (y < y3)
        {

	        HLineZT(p,x,y,z,s,t,x0,y,z0,s0,t0,Face->GetTexture(),Face->GetTexX(),Face->GetTexY());
			
            y++;
			
			t += dt;
			t0 += d0t;
			
			s += ds;
			s0 += d0s;
			
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }
    else
    {
        while (y < y3)
        {
	        
			HLineZT(p,x0,y,z0,s0,t0,x,y,z,s,t,Face->GetTexture(),Face->GetTexX(),Face->GetTexY());
			
            y++;
			
			t += dt;
			t0 += d0t;
			
			s += ds;
			s0 += d0s;
			
			z += dz;
			z0 += d0z;
            x  += d;
            x0 += d0;
        }
    }


	}


}
